# This file is Copyright 2019 Volatility Foundation and licensed under the Volatility Software License 1.0
# which is available at https://www.volatilityfoundation.org/license/vsl-v1.0
#

from typing import List, Tuple, Optional
import logging
from volatility3.framework import interfaces
from volatility3.framework import renderers, symbols
from volatility3.framework.configuration import requirements
from volatility3.framework.objects import utility
from volatility3.framework.renderers import format_hints
from volatility3.plugins.linux import pslist

vollog = logging.getLogger(__name__)


class Malfind(interfaces.plugins.PluginInterface):
    """Lists process memory ranges that potentially contain injected code."""

    _required_framework_version = (2, 0, 0)
    _version = (1, 0, 4)

    @classmethod
    def get_requirements(cls) -> List[interfaces.configuration.RequirementInterface]:
        return [
            requirements.ModuleRequirement(
                name="kernel",
                description="Linux kernel",
                architectures=["Intel32", "Intel64"],
            ),
            requirements.VersionRequirement(
                name="pslist", component=pslist.PsList, version=(4, 0, 0)
            ),
            requirements.ListRequirement(
                name="pid",
                description="Filter on specific process IDs",
                element_type=int,
                optional=True,
            ),
            requirements.IntRequirement(
                name="dump-size",
                description="Amount of bytes to dump for each dirty region/page found - Default 64 bytes",
                optional=True,
                default=64,
            ),
            requirements.BooleanRequirement(
                name="dump-page",
                description="Dump each dirty page and content - Default off",
                optional=True,
                default=False,
            ),
        ]

    def _list_injections(
        self, task
    ) -> Tuple[interfaces.objects.ObjectInterface, Optional[str], bytes]:
        """Generate memory regions for a process that may contain injected
        code."""

        proc_layer_name = task.add_process_layer()
        if not proc_layer_name:
            return None

        proc_layer = self.context.layers[proc_layer_name]

        dump_size = self.config["dump-size"]

        # Dumping page defaults to off, as in case a whole r-xp region is dirty
        # this would likely dump 1000's of pages which might not always be wise nor necessary

        dump_page = self.config["dump-page"]

        for vma in task.mm.get_vma_iter():
            vma_name = vma.get_name(self.context, task)
            vollog.debug(
                f"Injections : processing PID {task.pid} : VMA {vma_name} : {hex(vma.vm_start)}-{hex(vma.vm_end)}"
            )

            # If is_suspicious returns true, this means at least one page
            # in the region is dirty. If dump_page is true, then we dump
            # all dirty pages

            if vma.is_suspicious(proc_layer) and vma_name != "[vdso]":
                malicious_pages = vma.get_malicious_pages(proc_layer)
                offset = 0
                if dump_page:
                    # Dumping each dirty page
                    for page_addr in malicious_pages:
                        offset = page_addr - vma.vm_start
                        data = proc_layer.read(page_addr, dump_size, pad=True)
                        yield vma, f"{vma_name}, page address: {page_addr:#x}, offset: {offset:#x}", data, offset
                else:
                    # Original behaviour - Dump the start of the region (not necessarily matching the dirty page)
                    data = proc_layer.read(vma.vm_start, dump_size, pad=True)
                    yield vma, vma_name, data, offset

    def _generator(self, tasks):
        # determine if we're on a 32 or 64 bit kernel
        vmlinux = self.context.modules[self.config["kernel"]]
        is_32bit_arch = not symbols.symbol_table_is_64bit(
            context=self.context, symbol_table_name=vmlinux.symbol_table_name
        )

        for task in tasks:
            process_name = utility.array_to_string(task.comm)

            for vma, vma_name, data, offset in self._list_injections(task):
                if is_32bit_arch:
                    architecture = "intel"
                else:
                    architecture = "intel64"

                disasm = renderers.Disassembly(
                    data, vma.vm_start + offset, architecture
                )

                yield (
                    0,
                    (
                        task.pid,
                        process_name,
                        format_hints.Hex(vma.vm_start),
                        format_hints.Hex(vma.vm_end),
                        vma_name or renderers.NotAvailableValue(),
                        vma.get_protection(),
                        format_hints.HexBytes(data),
                        disasm,
                    ),
                )

    def run(self):
        filter_func = pslist.PsList.create_pid_filter(self.config.get("pid", None))

        return renderers.TreeGrid(
            [
                ("PID", int),
                ("Process", str),
                ("Start", format_hints.Hex),
                ("End", format_hints.Hex),
                ("Path", str),
                ("Protection", str),
                ("Hexdump", format_hints.HexBytes),
                ("Disasm", renderers.Disassembly),
            ],
            self._generator(
                pslist.PsList.list_tasks(
                    self.context, self.config["kernel"], filter_func=filter_func
                )
            ),
        )
